/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MN2WS_PCM_H__
#define MN2WS_PCM_H__

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/timer.h>
#include <linux/platform_device.h>

#include <sound/core.h>
#include <sound/initval.h>
#include <sound/asound.h>
#include <sound/pcm.h>
#include <sound/hwdep.h>
#include <sound/control.h>

#include "mn2ws-pcm_util.h"

#define MN2WS_PCM_PROC_ENTRY     "driver/mn2ws_pcm"

enum MN2WS_PCMIF_TYPE {
	MN2WS_PCMIF_UNKNOWN, 
	MN2WS_PCMIF_PLAYBACK, 
	MN2WS_PCMIF_CAPTURE, 
};

struct mn2ws_pcm_dev;

struct mn2ws_pcm_chip {
	//chip record should have the field to hold the card pointer at least
	struct snd_card *card;
	
	//pcm playback/capture device
	struct snd_pcm *pcm;
	
	//hwdep device
	struct snd_hwdep *hwdep;
	
	//pcm substream
	struct snd_pcm_substream *substream;
	
	//device
	struct mn2ws_pcm_dev *device;
};

struct mn2ws_pcm_pcmif_desc {
	int enabled;
	
	unsigned long phys_addr_buf;
	//size of HW buffer of requiring to the kernel
	size_t size_buf;
	//to fix the size of using HW buffer
	size_t size_use_fix;
	int (*init)(struct mn2ws_pcm_dev *dev);
	int (*term)(struct mn2ws_pcm_dev *dev);
	int (*hardware)(struct mn2ws_pcm_dev *dev, struct snd_pcm_hardware *h);
	int (*setup)(struct mn2ws_pcm_dev *dev);
	int (*mute)(struct mn2ws_pcm_dev *dev);
	int (*start)(struct mn2ws_pcm_dev *dev);
	int (*stop)(struct mn2ws_pcm_dev *dev);
	unsigned long (*get_hwptr)(struct mn2ws_pcm_dev *dev);
	void (*set_devptr)(struct mn2ws_pcm_dev *dev, unsigned long ptr);
	int (*wait_hwevent)(struct mn2ws_pcm_dev *dev);
	ssize_t (*copy)(struct mn2ws_pcm_dev *dev, struct ringbuf *r, size_t s);
	ssize_t (*silence)(struct mn2ws_pcm_dev *dev, size_t s);
};

struct mn2ws_pcm_hwd_desc {
	int enabled;
	
	int (*force_aout)(struct mn2ws_pcm_dev *dev);
};

struct mn2ws_pcm_desc {
	struct resource *res_regs;
	size_t n_res_regs;
	
	struct mn2ws_pcm_pcmif_desc play;
	struct mn2ws_pcm_pcmif_desc cap;
	struct mn2ws_pcm_hwd_desc hwd;
	
	void *private_data;
};

struct mn2ws_pcm_pcmif {
	struct mn2ws_pcm_pcmif_desc *desc;
	
	enum MN2WS_PCMIF_TYPE type;
	int (* thread_main)(void *data);
	
	spinlock_t spin;
	wait_queue_head_t q;
	int start_trigger, stop_trigger;
	int silent_mode;
	//physical address of HW buffer
	dma_addr_t paddr_buf;
	//virtual address of HW buffer
	unsigned char *buf;
	//size of using HW buffer
	// <------ desc.size_buf  ------>
	// <-- size_buf_use -->
	// | sound data       | not use |
	size_t size_buf_use;
	struct task_struct *thread;
	struct ringbuf hw;
	struct ringbuf alsa;
};

struct mn2ws_pcm_hwd {
	struct mn2ws_pcm_hwd_desc *desc;
	
	spinlock_t spin;
	
	int scale_pcm;
	int scale_dec;
};

struct mn2ws_pcm_dev {
	struct semaphore sem_dev;
	
	int ch;
	int card_index;
	char card_xid[16];
	struct mn2ws_pcm_desc *desc;
	struct snd_card *card;
	struct mn2ws_pcm_chip *chip;
	struct snd_pcm *pcm;
	struct snd_hwdep *hwdep;
	
	struct mem_mapping *map_regs;
	
	struct platform_device *plat_dev;
	
	struct mn2ws_pcm_pcmif play;
	struct mn2ws_pcm_pcmif cap;
	struct mn2ws_pcm_hwd hwd;
	
	void *private_data;
};


//for test mode
extern int mn2ws_pcm_test_mode;


//pcmif common(playback/capture)
const char *mn2ws_pcm_pcmif_get_type(struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_init(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_exit(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_alloc_hw_buffer(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_free_hw_buffer(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_alloc_bounce_buffer(struct snd_pcm_substream *substream, size_t size, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_free_bounce_buffer(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif);

int mn2ws_pcm_pcmif_reset_hw_ringbuf(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif);
int mn2ws_pcm_pcmif_reset_bounce_ringbuf(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif);

//pcmif common(soc)
int mn2ws_pcm_pcmif_hardware_16k_16b_1ch(struct mn2ws_pcm_dev *d, struct snd_pcm_hardware *h);
int mn2ws_pcm_pcmif_hardware_48k_16b_2ch(struct mn2ws_pcm_dev *d, struct snd_pcm_hardware *h);
int mn2ws_pcm_pcmif_hardware_48k_32b_2ch(struct mn2ws_pcm_dev *d, struct snd_pcm_hardware *h);
int mn2ws_pcm_pcmif_hrtimeout(struct mn2ws_pcm_dev *d);
ssize_t mn2ws_pcm_play_copy_to_hw(struct mn2ws_pcm_dev *d, struct ringbuf *r, size_t s);
ssize_t mn2ws_pcm_play_silence_to_hw(struct mn2ws_pcm_dev *d, size_t s);
ssize_t mn2ws_pcm_cap_copy_to_alsa(struct mn2ws_pcm_dev *d, struct ringbuf *r, size_t s);
ssize_t mn2ws_pcm_cap_silence_to_alsa(struct mn2ws_pcm_dev *d, size_t s);

//pcmif common(debug)
void mn2ws_pcm_dbg_pcmif_runtime(struct snd_pcm_substream *substream, struct mn2ws_pcm_pcmif *pcmif);
void mn2ws_pcm_dbg_pcmif_hwbuf(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
void mn2ws_pcm_dbg_pcmif_ringbuf(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);
void mn2ws_pcm_dbg_pcmif_ringbuf_maxmin(struct mn2ws_pcm_dev *d, struct mn2ws_pcm_pcmif *pcmif);


//playback
extern struct snd_pcm_ops mn2ws_pcm_playback_ops;
extern struct snd_pcm_hardware mn2ws_pcm_playback_hw;

int mn2ws_pcm_play_thread(void *data);


//capture
extern struct snd_pcm_ops mn2ws_pcm_capture_ops;
extern struct snd_pcm_hardware mn2ws_pcm_capture_hw;

int mn2ws_pcm_cap_thread(void *data);


//hwdep
#define SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_PCM    _IOR('H', 0x80, int *)
#define SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_PCM    _IOW('H', 0x81, int *)
#define SNDRV_HWDEP_IOCTL_MN2WS_GET_MUTE_DEC    _IOR('H', 0x82, int *)
#define SNDRV_HWDEP_IOCTL_MN2WS_SET_MUTE_DEC    _IOW('H', 0x83, int *)

#define SNDRV_HWDEP_IOCTL_MN2WS_FORCE_AOUT      _IO('H', 0xf0)

#define SNDRV_MN2WS_SCALE_MAX    8
#define SNDRV_MN2WS_SCALE_MIN    0

extern struct snd_hwdep_ops mn2ws_pcm_hwdep_ops;

int mn2ws_pcm_hwdep_init(struct mn2ws_pcm_dev *d);
int mn2ws_pcm_hwdep_exit(struct mn2ws_pcm_dev *d);


//control
extern struct snd_kcontrol_new mn2ws_pcm_mixer_ops[];
extern int mn2ws_pcm_mixer_num;

//common utils
int mn2ws_pcm_init_module(struct mn2ws_pcm_desc descs[]);
int mn2ws_pcm_exit_module(struct mn2ws_pcm_desc descs[]);


//for each LSI
const char *get_pcm_dev_name(void);
int get_pcm_dev_count(void);

#endif //MN2WS_PCM_H__

